#include "CryptoApp.hpp"
#include <iostream>
#include <stdexcept>

using namespace std;

// Constructor per defecte de CryptoApp
// PRE: Cert
// POST: Crea una aplicació buida
CryptoApp::CryptoApp() {
}

// Destructor
// PRE: Cert
// POST: Esborra automàticament els objectes locals
CryptoApp::~CryptoApp() {}

// PRE: Cert
// POST: Retorna l'índex de l'asset amb assetSymbol, o -1 si no existeix
int CryptoApp::findAssetIndexBySymbol(const string& assetSymbol) const {
    
    int x;
	bool found = false;
	unsigned int i = 0;
	// Invariant: i < vec_assets.size() i si found == false implica que assetSymbol no està en vec_assets[0..i-1]
	while (found == false and i<vec_assets.size()) {
		if (vec_assets[i].getAssetSymbol() == assetSymbol) {
			x = i;
			found = true;
		}
		else {
			++i;
		}
	}
	
	if (found == false) {
		x = -1;
	}
	return x;
}

// PRE: cap
// POST: Retorna l'índex de l'asset amb assetId, o -1 si no existeix
int CryptoApp::findAssetIndexById(int assetId) const {
    
    int x;
	bool found = false;
	unsigned int i = 0;
	// Invariant: i < vec_assets.size() i si found == false implica que assetId no està en vec_assets[0..i-1]
	while (found == false and i<vec_assets.size()) {
		if (vec_assets[i].getId() == assetId) {
			x = i;
			found = true;
		}
		else {
			++i;
		}
	}
	
	if (found == false) {
		x = -1;
	}
	return x;
}

// PRE: cap
// POST: Retorna l'índex del user amb email, o -1 si no existeix
int CryptoApp::findUserIndexByEmail(const string& email) const {
    
    int x;
	bool found = false;
	unsigned int i = 0;
	// Invariant: i < vec_users.size() i si found == false implica que email no està en vec_users[0..i-1]
	while (found == false and i<vec_users.size()) {
		if (vec_users[i].getEmail() == email) {
			x = i;
			found = true;
		}
		else {
			++i;
		}
	}
	
	if (found == false) {
		x = -1;
	}
	return x;
}

// PRE: cap
// POST: Retorna l'índex del user amb name, o -1 si no existeix
int CryptoApp::findUserIndexByName(const string& name) const {
    
    int x;
	bool found = false;
	unsigned int i = 0;
	// Invariant: i < vec_users.size() i si found == false implica que name no està en vec_users[0..i-1]
	while (found == false and i<vec_users.size()) {
		if (vec_users[i].getName() == name) {
			x = i;
			found = true;
		}
		else {
			++i;
		}
	}
	
	if (found == false) {
		x = -1;
	}
	return x;
}

// PRE: cap
// POST: Retorna l'índex del holding amb (userId, assetId), o -1 si no existeix
int CryptoApp::findHoldingIndex(int userId, int assetId) const {
    
    int x;
	bool found = false;
	unsigned int i = 0;
	// Invariant: i < vec_holdings.size() i si found == false implica que (userId, assetId) no està en vec_holdings[0..i-1]
	while (found == false and i<vec_holdings.size()) {
		if (vec_holdings[i].getUserId() == userId and vec_holdings[i].getAssetId() == assetId) {
			x = i;
			found = true;
		}
		else {
			++i;
		}
	}
	
	if (found == false) {
		x = -1;
	}
	return x;
}

// PRE: assetId vàlid
// POST: Retorna l'últim preu de l'asset, o -1 si no té preus
double CryptoApp::getLastPrice(int assetId) const {
    double x = -1;
    
    int assetIndex = findAssetIndexById(assetId);
    
    if (assetIndex != -1) {
        const vector<Price>& prices = vec_assets[assetIndex].getPrices();
        
        if (!prices.empty()) {
            x = prices[prices.size() - 1].getPrice();
        }
    }
    
    return x;
}


// PRE: email no buit
// POST: Retorna true si l'email té format bàsic vàlid (.+@.+)
bool CryptoApp::isValidEmail(const string& email) const {
    
    bool validEmail = true;
    
    if (email.empty()) {
        validEmail = false;
    }
    
    unsigned int atPos = -1;
    bool found = false;
    unsigned int i = 0;
    
	// Invariant: i < email.size() i si found == false implica que '@' no està en email[0..i-1]
    while (!found and i<email.size()) {
		if (email[i] == '@') {
			atPos = i;
			found = true;
		}
		else {
			++i;
		}
	}
    
    if (atPos <= 0 or atPos >= email.size() - 1) {
        validEmail = false;
    }
    
    return validEmail;
}

// PRE: cap
// POST: Crea l'actiu EUR si no existeix
void CryptoApp::ensureEURAsset() {
    if (findAssetIndexBySymbol("EUR") == -1) {
        Asset eurAsset("EUR", "Euro", 2);
        vec_assets.push_back(eurAsset);
    }
}

// CRUD d'Assets

// PRE: assetSymbol no buit, decimals en [0,18]
// POST: Crea un nou asset. Retorna l'id de l'asset creat
int CryptoApp::createAsset(const string& assetSymbol, const string& name, int decimals) {
   
    if (findAssetIndexBySymbol(assetSymbol) != -1) {
        throw runtime_error("Error al crear Actiu " + assetSymbol + ". L'Actiu ja existeix");
    }
    
    if (decimals < 0 or decimals > 18) {
        throw runtime_error("Error al crear l'actiu. El número de decimals ha d'estar entre 0 i 18");
    }
    
    Asset newAsset(assetSymbol, name, decimals);
    vec_assets.push_back(newAsset);
    
    return newAsset.getId();
}

// PRE: cap
// POST: Llista tots els assets per ordre d'alta
void CryptoApp::listAssets() const {
    cout << "--------------------" << endl;
    // Invariant: i < vec_assets.size() i s'han mostrat els assets vec_assets[0..i-1]
    for (unsigned int i = 0; i < vec_assets.size(); i++) {
        cout << vec_assets[i] << endl;
    }
    cout << "--------------------" << endl;
}

// CRUD de Prices

// PRE: assetSymbol existeix, dateISO en format YYYY-MM-DD, price > 0
// POST: Crea un nou preu per l'asset. Retorna l'id del preu
int CryptoApp::createPrice(const string& assetSymbol, const string& dateISO, double price) {
  
    int assetIndex = findAssetIndexBySymbol(assetSymbol);
    if (assetIndex == -1) {
        throw runtime_error("Error al crear Preu de " + assetSymbol + " a " + dateISO + ". L'actiu no existeix");
    }
    
    if (assetSymbol == "EUR") {
        throw runtime_error("Error al crear Preu de EUR a " + dateISO + ". No es permet crear preus per a EUR");
    }

    if (!isValidDateISO(dateISO)) {
        throw runtime_error("Error al crear Preu de " + assetSymbol + " a " + dateISO + ". La data incorrecte en format, s'espera YYYY-MM-DD");
    }
    
    if (price <= 0) {
        throw runtime_error("Error al crear Preu de " + assetSymbol + " a " + dateISO + ". El preu està en format incorrecte, ha de ser major que 0");
    }
    
    const vector<Price>& prices = vec_assets[assetIndex].getPrices();
    // Invariant: i < prices.size() i dateISO no està en prices[0..i-1]
    for (unsigned int i = 0; i < prices.size(); i++) {
        if (prices[i].getDateISO() == dateISO) {
            throw runtime_error("Error al crear Preu de " + assetSymbol + " a " + dateISO + ". Ja s'ha introduït un preu per a aquest actiu i data");
        }
    }
    
    
    int assetId = vec_assets[assetIndex].getId();
    Price newPrice(assetId, dateISO, price);
    vec_assets[assetIndex].addPrice(newPrice);
    
    return newPrice.getId();
}

// PRE: cert
// POST: Llista tots els preus de l'asset per ordre d'alta
void CryptoApp::listPricesOfAsset(const string& assetSymbol) const {
    int assetIndex = findAssetIndexBySymbol(assetSymbol);
    if (assetIndex == -1) {
        throw runtime_error("Error al mostrar preus. Actiu " + assetSymbol + " no existeix");
    }
    
    const vector<Price>& prices = vec_assets[assetIndex].getPrices();
    
    
    if (prices.empty()) {
        throw runtime_error("Error al mostrar preus. Actiu " + assetSymbol + " no té un preu actual");
    }
    
   
    cout << "--------------------" << endl;
    // Invariant: i < prices.size() i s'han mostrat els preus prices[0..i-1]
    for (unsigned int i = 0; i < prices.size(); i++) {
        cout << assetSymbol << " " << prices[i].getDateISO() << " " << prices[i].getPrice() << endl;
    }
    cout << "--------------------" << endl;
}

// PRE: assetSymbol existeix
// POST: Mostra l'últim preu de l'asset
void CryptoApp::showCurrentPrice(const string& assetSymbol) const {
    int assetIndex = findAssetIndexBySymbol(assetSymbol);
    if (assetIndex == -1) {
        throw runtime_error("Error al mostrar preu actual de l'actiu " + assetSymbol + ". Actiu no existent");
    }
    
    const vector<Price>& prices = vec_assets[assetIndex].getPrices();
    if (prices.empty()) {
        throw runtime_error("Error al mostrar preu actual de l'actiu " + assetSymbol + ". L'actiu no té historic de preus.");
    }
    
    cout << "--------------------" << endl;
    const Price& lastPrice = prices[prices.size() - 1];
    cout << assetSymbol << " " << lastPrice.getDateISO() << " " << lastPrice.getPrice() << endl;
    cout << "--------------------" << endl;
}

// CRUD d'Users

// PRE: name no buit (<=50), email vàlid (<=50), password (<=50)
// POST: Crea un nou usuari. Retorna l'id de l'usuari
int CryptoApp::createUser(const string& name, const string& email, const string& password) {
    
    if (vec_users.empty()) {
		ensureEURAsset();
	}
    
    if (!isValidEmail(email) or email.size() > 50) {
        throw runtime_error("Error al crear Usuari. Email incorrecte");
    }
    
    
    if (findUserIndexByEmail(email) != -1) {
        throw runtime_error("Error al crear Usuari " + name);
    }
    
    
    User newUser(name, email, password);
    vec_users.push_back(newUser);
    
    return newUser.getId();
}

// PRE: cap
// POST: Llista tots els usuaris per ordre d'alta
void CryptoApp::listUsers() const {
    cout << "--------------------" << endl;
    // Invariant: i < vec_users.size() i s'han mostrat els usuaris vec_users[0..i-1]
    for (unsigned int i = 0; i < vec_users.size(); i++) {
        cout << vec_users[i] << endl;
    }
    cout << "--------------------" << endl;
}

// CRUD d'Addresses

// PRE: userName existeix, assetSymbol existeix, address i label no buits
// POST: Crea una nova address per l'usuari. Retorna l'id de l'address
int CryptoApp::createAddress(const string& userName, const string& assetSymbol, const string& address, const string& label) {
    
    int userIndex = findUserIndexByName(userName);
    if (userIndex == -1) {
        throw runtime_error("Error al crear Adreça. L'usuari " + userName + " no existeix");
    }
    
    int userId = vec_users[userIndex].getId();
    if (vec_users[userIndex].hasAddress(assetSymbol, address)) {
        throw runtime_error("Error al crear Adreça. Adreça duplicada");
    }
    
    
    Address newAddress(userId, assetSymbol, address, label);
    vec_users[userIndex].addAddress(newAddress);
    
    return newAddress.getId();
}

// PRE: userName existeix
// POST: Llista totes les addresses de l'usuari
void CryptoApp::listAddressesOfUser(const string& userName) const {
    int userIndex = findUserIndexByName(userName);
    if (userIndex == -1) {
        throw runtime_error("Error al mostrar Adreces. Usuari " + userName + " no existeix");
    }
	
	cout << "--------------------" << endl;
	
    const vector<Address>& addresses = vec_users[userIndex].getAddresses();
    // Invariant: i < addresses.size() i s'han mostrat les addresses addresses[0..i-1]
    for (unsigned int i = 0; i < addresses.size(); i++) {
        cout << addresses[i] << endl;
    }
    cout << "--------------------" << endl;
}

// Operacions de negoci

// PRE: userName existeix, quantity > 0
// POST: Afegeix quantity euros al holding EUR de l'usuari
void CryptoApp::addEuros(const string& userName, double quantity) {
    
    int userIndex = findUserIndexByName(userName);
    if (userIndex == -1) {
        throw runtime_error("Error al afegir Euros a l'usuari " + userName + ". L'usuari no existeix");
    }
    
    
    if (quantity <= 0) {
        throw runtime_error("Error al afegir Euros a l'usuari " + userName + ". Quantitat incorrecte");
    }
    
    ensureEURAsset();
    
    int eurIndex = findAssetIndexBySymbol("EUR");
    int userId = vec_users[userIndex].getId();
    int eurId = vec_assets[eurIndex].getId();
    
    
    int holdingIndex = findHoldingIndex(userId, eurId);
    
    if (holdingIndex == -1) {
        
        Holding newHolding(userId, eurId, quantity, 1.0);
        vec_holdings.push_back(newHolding);  
    } 
    else {
        
        vec_holdings[holdingIndex].buy(quantity, 1.0);
    }
}

// PRE: userName existeix, quantity > 0
// POST: Retira quantity euros (o tot el saldo si quantity > saldo)
void CryptoApp::withdrawEuros(const string& userName, double quantity) {
    
    int userIndex = findUserIndexByName(userName);
    if (userIndex == -1) {
        throw runtime_error("Error al retirar Euros a l'usuari " + userName + ". L'usuari no existeix");
    }
    
    
    if (quantity <= 0) {
        throw runtime_error("Error al retirar Euros a l'usuari " + userName + ". Quantitat incorrecte");
    }
    
    ensureEURAsset();
    
    int eurIndex = findAssetIndexBySymbol("EUR");
    int userId = vec_users[userIndex].getId();
    int eurId = vec_assets[eurIndex].getId();
    
    
    int holdingIndex = findHoldingIndex(userId, eurId);
    
    if (holdingIndex != -1) {
        
        double currentAmount = vec_holdings[holdingIndex].getAmount();
		double toWithdraw = (quantity > currentAmount) ? currentAmount : quantity;
    
		vec_holdings[holdingIndex].sell(toWithdraw);
    
		cout << toWithdraw << " Euros retirats correctament a l'usuari " << userName << endl;
    }  
}

// PRE: userName i assetSymbol existeixen, quantity > 0
// POST: Compra quantity unitats de l'asset al preu actual
void CryptoApp::buy(const string& userName, const string& assetSymbol, double quantity) {
    
    int userIndex = findUserIndexByName(userName);
    if (userIndex == -1) {
        throw runtime_error("Error al comprar " + assetSymbol + ". Usuari " + userName + " no existeix");
    }
    
    if (quantity <= 0) {
        throw runtime_error("Error al comprar " + assetSymbol + ". La quantitat ha de ser major a 0");
    }
    
    int assetIndex = findAssetIndexBySymbol(assetSymbol);
    if (assetIndex == -1) {
        throw runtime_error("Error al comprar " + assetSymbol + ". Actiu " + assetSymbol + " no existeix");
    }
    
    int userId = vec_users[userIndex].getId();
    int assetId = vec_assets[assetIndex].getId();
    
    
    double currentPrice = getLastPrice(assetId);
    if (currentPrice <= 0) {
        throw runtime_error("Error al comprar " + assetSymbol + ". Actiu " + assetSymbol + " no té un preu actual");
    }
    
    double totalCost = quantity * currentPrice;
    
    ensureEURAsset();

    int eurIndex = findAssetIndexBySymbol("EUR");
    int eurId = vec_assets[eurIndex].getId();
    int eurHoldingIndex = findHoldingIndex(userId, eurId);
    
    if (eurHoldingIndex == -1 or vec_holdings[eurHoldingIndex].getAmount() < totalCost) {
        throw runtime_error("Error al comprar " + assetSymbol + ". L'usuari no disposa de fons suficients");
    }
    
    
    vec_holdings[eurHoldingIndex].sell(totalCost);
    
    
    int assetHoldingIndex = findHoldingIndex(userId, assetId);
    
    if (assetHoldingIndex == -1) {
        Holding newHolding(userId, assetId, quantity, currentPrice);
        vec_holdings.push_back(newHolding);
    } 
    else {
        vec_holdings[assetHoldingIndex].buy(quantity, currentPrice);
    }
}

// PRE: userName i assetSymbol existeixen, quantity > 0
// POST: Ven quantity unitats (o totes si quantity > amount)
void CryptoApp::sell(const string& userName, const string& assetSymbol, double quantity) {
    
    int userIndex = findUserIndexByName(userName);
    if (userIndex == -1) {
        throw runtime_error("Error al vendre " + assetSymbol + ". Usuari " + userName + " no existeix");
    }
    
    if (quantity <= 0) {
        throw runtime_error("Error al vendre " + assetSymbol + ". La quantitat ha de ser major a 0");
    }
    
    int assetIndex = findAssetIndexBySymbol(assetSymbol);
    if (assetIndex == -1) {
        throw runtime_error("Error al vendre " + assetSymbol + ". Actiu " + assetSymbol + " no existeix");
    }
    
    int userId = vec_users[userIndex].getId();
    int assetId = vec_assets[assetIndex].getId();
    
    
    int holdingIndex = findHoldingIndex(userId, assetId);
    if (holdingIndex == -1) {
        throw runtime_error("Error al vendre " + assetSymbol + ". L'usuari no posseeix l'actiu " + assetSymbol);
    }
    
    
    double currentPrice = getLastPrice(assetId);
    if (currentPrice <= 0) {
        throw runtime_error("Error al vendre " + assetSymbol + ". Actiu " + assetSymbol + " no té un preu actual");
    }
    
    
    double currentAmount = vec_holdings[holdingIndex].getAmount();
    double toSell = (quantity > currentAmount) ? currentAmount : quantity;
    
   
    double totalIncome = toSell * currentPrice;
    
    
    vec_holdings[holdingIndex].sell(toSell);
    
    ensureEURAsset();

    
    int eurIndex = findAssetIndexBySymbol("EUR");
    int eurId = vec_assets[eurIndex].getId();
    int eurHoldingIndex = findHoldingIndex(userId, eurId);
    
    if (eurHoldingIndex == -1) {
        Holding newHolding(userId, eurId, totalIncome, 1.0);
        vec_holdings.push_back(newHolding);
    } 
    else {
        vec_holdings[eurHoldingIndex].buy(totalIncome, 1.0);
    }
    
    cout << "Venda de " << toSell << " unitats registrada per " << userName << " en " << assetSymbol << endl;
}

// Consultes

// PRE: dateISO en format YYYY-MM-DD
// POST: Retorna true si la data té format vàlid i valors correctes
bool CryptoApp::isValidDateISO(const string& dateISO) const {
    
    bool correct = true;

    if (dateISO.size() != 10) {
        correct = false;
    }
    
    
    if (dateISO[4] != '-' || dateISO[7] != '-') {
        correct = false;
    }
    
    
    // Invariant: i < 10 i correct és fals si existeix un dígit invàlid en dateISO[0..i-1] (excepte posicions 4 i 7)
    for (int i = 0; i < 10; ++i) {
		if (i != 4) {
			if (i != 7) {
				if (dateISO[i] < '0' or dateISO[i] > '9') {
					correct = false;
				}
			}
		}
	}
    
    
    int mes = (dateISO[5] - '0') * 10 + (dateISO[6] - '0');
    int dia = (dateISO[8] - '0') * 10 + (dateISO[9] - '0');
    
    
    if (mes < 1 or mes > 12) {
        correct = false;
    }
    
    
    if (dia < 1 or dia > 31) {
        correct = false;
    }

    return correct;
}

// PRE: userName existeix
// POST: Mostra la cartera completa de l'usuari amb valoració
void CryptoApp::evaluatePortfolio(const string& userName) const {
    int userIndex = findUserIndexByName(userName);
    if (userIndex == -1) {
        throw runtime_error("Error al mostrar tenencies per l'usuari: " + userName);
    }
    
    int userId = vec_users[userIndex].getId();
    
    cout << "--------------------" << endl;
    
    double totalAvgCost = 0;
    double totalCurrentValue = 0;
    double totalProfit = 0;
    
    // Invariant: i < vec_holdings.size() i s'han processat els holdings vec_holdings [0..i-1] per calcular totalAvgCost, totalCurrentValue i totalProfit
    for (unsigned int i = 0; i < vec_holdings.size(); i++) {
        
        if (vec_holdings[i].getUserId() == userId) {
            
            int assetId = vec_holdings[i].getAssetId();
            int assetIndex = findAssetIndexById(assetId);
            
            string symbol = vec_assets[assetIndex].getAssetSymbol();
            double amount = vec_holdings[i].getAmount();
            double avgCost = vec_holdings[i].getAvgCost();
            
            double actualPrice;
            
            if (symbol == "EUR") {
                actualPrice = 1.0;
            } 
            else {
                actualPrice = getLastPrice(assetId);
                if (actualPrice <= 0) {
                    actualPrice = 0;
                }
            }
            
            cout << symbol << " - amount: " << amount;
            if (symbol != "EUR") {
                cout << " - avgCost: " << avgCost << " - preuActual: " << actualPrice;
            }
            cout << endl;
            
            totalAvgCost += amount * avgCost;
            totalCurrentValue += vec_holdings[i].getCurrentValue(actualPrice);
            totalProfit += vec_holdings[i].getProfit(actualPrice);
        }
    }
    
    cout << "Total avgCost: " << totalAvgCost << " - Total preuActual: " << totalCurrentValue << " - Guany total: " << totalProfit << endl;
    cout << "--------------------" << endl;
}



// Mètode d'eliminació d'usuari 

// PRE: name és el nom d'usuari a eliminar
// POST: Elimina l'usuari i tots els seus holdings associats de manera eficient
void CryptoApp::deleteUser(const string& name) {
    
    // Aquest és el mètode que heu d'implementar
}

