#ifndef CRYPTOAPP_HPP
#define CRYPTOAPP_HPP

#include <vector>
#include <string>
#include <iostream>
#include "Asset.hpp"
#include "Price.hpp"
#include "User.hpp"
#include "Address.hpp"
#include "Holding.hpp"

using namespace std;

class CryptoApp {
private:
    //Vector d'assets.
    vector<Asset> vec_assets;

    //Vector d'usuaris.
    vector<User> vec_users;

    //Vector de holdings.
    vector<Holding> vec_holdings;
        
    // PRE: cert
    // POST: Retorna l'índex de l'asset amb assetSymbol, o -1 si no existeix
    int findAssetIndexBySymbol(const string& assetSymbol) const;
    
    // PRE: cert
    // POST: Retorna l'índex de l'asset amb assetId, o -1 si no existeix
    int findAssetIndexById(int assetId) const;
    
    // PRE: cert
    // POST: Retorna l'índex del user amb email, o -1 si no existeix
    int findUserIndexByEmail(const string& email) const;
    
    // PRE: cert
    // POST: Retorna l'índex del user amb name, o -1 si no existeix
    int findUserIndexByName(const string& name) const;
    
    // PRE: cert
    // POST: Retorna l'índex del holding amb (userId, assetId), o -1 si no existeix
    int findHoldingIndex(int userId, int assetId) const;
    
    // PRE: assetId vàlid
    // POST: Retorna l'últim preu de l'asset, o -1 si no té preus
    double getLastPrice(int assetId) const;
    
    // PRE: email no buit
    // POST: Retorna true si l'email té format bàsic vàlid (.+@.+)
    bool isValidEmail(const string& email) const;
    
    // PRE: dateISO en format YYYY-MM-DD
    // POST: Retorna true si la data té format vàlid i valors correctes
    bool isValidDateISO(const string& dateISO) const;

    // PRE: cert
    // POST: Crea l'actiu EUR si no existeix
    void ensureEURAsset();

public:
    // Constructor
    // PRE: cert
    // POST: Crea una aplicació buida i assegura que existeix l'actiu EUR
    CryptoApp();
    
    // Destructor
    // PRE: cert
    // POST: Elimina els recursos
    ~CryptoApp();
    
    // CRUD d'Assets
    
    // PRE: assetSymbol no buit, decimals en [0,18]
    // POST: Crea un nou asset. Retorna l'id de l'asset creat
    //       Llança excepció si assetSymbol ja existeix o decimals fora de rang
    int createAsset(const string& assetSymbol, const string& name, int decimals);
    
    // PRE: cert
    // POST: Llista tots els assets per ordre d'alta
    void listAssets() const;
    
    // CRUD de Prices
    
    // PRE: assetSymbol existeix, dateISO en format YYYY-MM-DD, price > 0
    // POST: Crea un nou preu per l'asset. Retorna l'id del preu
    //       Llança excepció si l'asset no existeix, price <= 0, o (assetSymbol, dateISO) duplicat
    int createPrice(const string& assetSymbol, const string& dateISO, double price);
    
    // PRE: assetSymbol existeix
    // POST: Llista tots els preus de l'asset per ordre d'alta
    //       Llança excepció si l'asset no existeix
    void listPricesOfAsset(const string& assetSymbol) const;
    
    // PRE: assetSymbol existeix
    // POST: Mostra l'últim preu de l'asset
    //       Llança excepció si l'asset no existeix o no té preus
    void showCurrentPrice(const string& assetSymbol) const;
    
    // CRUD d'Users
    
    // PRE: name no buit (<=50), email vàlid (<=50), password (<=50)
    // POST: Crea un nou usuari. Retorna l'id de l'usuari
    //       Llança excepció si email invàlid o ja existeix
    int createUser(const string& name, const string& email, const string& password);
    
    // PRE: cert
    // POST: Llista tots els usuaris per ordre d'alta
    void listUsers() const;
    
    // CRUD d'Addresses
    
    // PRE: userName existeix, assetSymbol existeix, address i label no buits
    // POST: Crea una nova address per l'usuari. Retorna l'id de l'address
    //       Llança excepció si usuari o asset no existeixen, o address duplicada
    int createAddress(const string& userName, const string& assetSymbol, const string& address, const string& label);
    
    // PRE: userName existeix
    // POST: Llista totes les addresses de l'usuari
    //       Llança excepció si l'usuari no existeix
    void listAddressesOfUser(const string& userName) const;
    
    // Operacions de negoci
    
    // PRE: userName existeix, quantity > 0
    // POST: Afegeix quantity euros al holding EUR de l'usuari
    //       Llança excepció si l'usuari no existeix o quantity <= 0
    void addEuros(const string& userName, double quantity);
    
    // PRE: userName existeix, quantity > 0
    // POST: Retira quantity euros (o tot el saldo si quantity > saldo)
    //       Llança excepció si l'usuari no existeix o quantity <= 0
    void withdrawEuros(const string& userName, double quantity);
    
    // PRE: userName i assetSymbol existeixen, quantity > 0, usuari té EUR suficients
    // POST: Compra quantity unitats de l'asset al preu actual
    //       Actualitza el holding de l'asset i redueix el holding EUR
    //       Llança excepció si falten fons, no hi ha preu, etc.
    void buy(const string& userName, const string& assetSymbol, double quantity);
    
    // PRE: userName i assetSymbol existeixen, quantity > 0, usuari té l'asset
    // POST: Ven quantity unitats (o totes si quantity > amount)
    //       Actualitza el holding de l'asset i augmenta el holding EUR
    //       Llança excepció si l'usuari no posseeix l'asset, no hi ha preu, etc.
    void sell(const string& userName, const string& assetSymbol, double quantity);
    
    // Consultes
    
    // PRE: userName existeix
    // POST: Mostra la cartera completa de l'usuari amb valoració
    //       Llança excepció si l'usuari no existeix
    void evaluatePortfolio(const string& userName) const;


    // PRE: name existeix
    // POST: Elimina l'usuari i tots els seus holdings associats.
    //       Llança excepció si l'usuari no existeix.
    void deleteUser(const string& name);



    //------------------------------------------------------------------------------------

    // --- MÈTODES AUXILIARS PER AL TEST (JUTGE) ---

    // PRE: Cert
    // POST: Retorna una referència constant al vector privat de holdings.
    const vector<Holding>& getHoldings() const {
        return vec_holdings;
    }
    
    // Helper per obtenir ID des del main
    // PRE: name no és buit.
    // POST: Retorna l'ID de l'usuari si existeix, o -1 si no es troba.
    int findUserIdByNamePublic(const string& name) const {
        int idx = findUserIndexByName(name);
        if (idx != -1) return vec_users[idx].getId();
        return -1;
    }

    //------------------------------------------------------------------------------------


};

#endif
